//*****************************************************************************************************
//
// The following Functions are used for Software Upgrading. They are located in the secure section.
// Beware: This can *break* and crash your program if used to erase or modify program space!
//
//*****************************************************************************************************
#include <p30F4011.h>
#include <libpic30.h>
#include "dsp.h"
#include "fft.h"
#include "Musicolour.h"
//
extern void blink(int);
extern void initRC5(void);
extern fractcomplex sigCmpx[FFT_BLOCK_LENGTH*2] __attribute__ ((section (".ydata, data, ymemory"), aligned (FFT_BLOCK_LENGTH*2*2)));
							/* 	Typically, the input signal to an FFT  	*/
							/* 	of an input signal. For this example, 	*/
							/* 	we will provide the input signal in an 	*/
							/* 	array declared in Y-data space.			*/

extern char buffer[NUM_DIGITS+3];						
// FIFO system vars
extern char keyBuffer[KEY_BUFFER_SIZE];	// keys pressed FIFO queue
extern int keyPtr;							// for keys pressed FIFO queue
extern int keyGetPtr;						// for keys pressed FIFO queue
extern int keyFull;				     	// number of keys in input key queue
// Flash Subsystem Variables
extern unsigned long hexaddress;
extern unsigned int* hexpointer;
extern unsigned int hexbytes;
extern unsigned int hextotal;
extern unsigned long hexrowaddress;
extern unsigned long hexprotectMin;
extern unsigned long hexprotectMax;
extern unsigned long hexprotectedBytes;
extern int ie0, ie1, ie2;
extern int rf6Mode;
extern void changeRf6Mode(int);
extern int writingMode;
extern int _EEDATA() arrayInDataEEPROM[];
char uartBuffer[UART_BUFFER_SIZE];
int uartFull, uartGetPtr, uartPtr, uartError;
//
//*****************************************************************************
// Optional UART code: in Secure Section
//*****************************************************************************
extern unsigned int addressErrorsW;
extern unsigned int addressErrorsR;
extern char addressErrorFlag;
extern char addressErrorReporting;
int zerorecords, tworecords, fourrecords;
int crcerrors;

void __attribute__((__section__(".SecureSection"))) securewait(void)
{
	DELAY_STRING;
}

void __attribute__((__section__(".SecureSection"))) disableAllInterrupts(void)
{
	ie0=IEC0;
	ie1=IEC1;
	ie2=IEC2;
	IEC0=0;	
	IEC1=0;
	IEC2=0;
}

void __attribute__((__section__(".SecureSection"))) enableAllInterrupts(void)
{
	IEC0=ie0;
	IEC1=ie1;
	IEC2=ie2;
}

//*****************************************************************************
// Uart Receive FIFO Functions and ISR
//*****************************************************************************
int __attribute__((__section__(".SecureSection"))) getFromUartFIFO(void)
{
	char g;
	if(uartFull>0){
		g=uartBuffer[uartGetPtr];
		uartGetPtr++;
		if(uartGetPtr>=UART_BUFFER_SIZE)uartGetPtr=0;
		uartFull--;
		} else return -1;
	return (0x00FF & g);
}

void __attribute__((__section__(".SecureSection"))) flushUartFIFO(void)
{
	uartPtr=0;
	uartGetPtr=0;
	uartFull=0;
	uartError=0;
}

void __attribute__((__section__(".SecureSection"))) addCharUartFIFO(char xx)
{
	if(uartFull<UART_BUFFER_SIZE){
			uartBuffer[uartPtr]=xx;
			uartPtr++;
			if(uartPtr>=UART_BUFFER_SIZE)uartPtr=0;
			uartFull++;
			uartError=0;
		} else uartError=1;
}
		
void __attribute__((__section__(".SecureSection"), interrupt, auto_psv)) _U1RXInterrupt(void)
{
	int k;
	while((uartError==0)&&(U1STAbits.URXDA==1)){
				k=U1RXREG;
				addCharUartFIFO(0x00FF & k);
	}
	while(uartError==1){
		flushUartFIFO();
	}		
	IFS0bits.U1RXIF=0;	
}
//*****************************************************************************
// Keys pressed FIFO functions
//*****************************************************************************
int __attribute__((__section__(".SecureSection"))) getKey(void)
{
	int g;
	if(keyFull>0){
	g=keyBuffer[keyGetPtr];
	keyGetPtr++;
	if(keyGetPtr>=KEY_BUFFER_SIZE)keyGetPtr=0;
	keyFull--;
	} else g=-1;
	return g;
}

void __attribute__((__section__(".SecureSection"))) flushKeys(void)
{
	keyPtr=0;
	keyGetPtr=0;
	keyFull=0;
	initRC5();
}

int __attribute__((__section__(".SecureSection"))) noKeys(void)
{
	if(keyFull>0)return FALSE; 
	else return TRUE;
}

int __attribute__((__section__(".SecureSection"))) waitForKey(void)
{
	while(keyFull<=0)securewait();
	return getKey();
}	

void __attribute__((__section__(".SecureSection"))) secureSetTimeOut(int y)
{
	T1CON=0x0000;
	TMR1=0x0000;							// clear T1
#if (DEBUG==0)
	PR1=y;									// the period to get around 1ms delay
#else
	PR1=1;
#endif
	T1CON=0x8030;							// control register setup to run off system clock at frequency FCY divided by 1:256
	IFS0bits.T1IF=0;
	IEC0bits.T1IE=0;
}

int __attribute__((__section__(".SecureSection"))) securetimeOut(void)
{
	if(IFS0bits.T1IF==1){ IFS0bits.T1IF=0; return TRUE; }
		else return FALSE;
}

int __attribute__((__section__(".SecureSection"))) securenoKeys(void)
{
	return TRUE;
}

unsigned int __attribute__((__section__(".SecureSection"))) initUart(void)
{
	U1MODE=0x0000;
	IEC0bits.U1RXIE=0;
	flushUartFIFO();
	U1STA=0x0000;
	changeRf6Mode(rf6Mode);
	IFS0bits.U1RXIF=0;
	IPC2|=(4<<4);			// the priority
	U1MODE=0x8000;
	U1STAbits.UTXEN=1;
	IEC0bits.U1RXIE=1;
	return UART_OK;
}

/*
unsigned int __attribute__((__section__(".SecureSection"))) disableUart(void)
{
	U1STA=0x0000;
	U1MODE=0x0000;
	return UART_OK;
}
*/

unsigned int __attribute__((__section__(".SecureSection"))) getFromUartPolling(void)
{
	unsigned int k, endit, x;

	secureSetTimeOut(SECURE_UART_TIMEOUT);
	endit=0;
	k=UART_ERROR;
	x=0;
	while((!securetimeOut())&&(endit==0)){
		if(U1STAbits.URXDA==1){ 
				x=(0x00FF & U1RXREG); 
				k=UART_OK; 
				endit=1; 
				}
		if(U1STAbits.OERR==1){ 
					U1STAbits.OERR=0; 
					k=UART_ERROR; 
					endit=1; 
				}
	}
	if(endit==0)k=UART_ERROR;
	if(k==UART_OK)return x; else return k;
}

unsigned int __attribute__((__section__(".SecureSection"))) getFromUart(void)
{
	int k;
	k=getFromUartFIFO();
	if(k!=-1)return (0x00FF & k); else return UART_ERROR;
}

unsigned int __attribute__((__section__(".SecureSection"))) putToUart(char Char)
{
	while(U1STAbits.TRMT==0)asm("nop");
	U1TXREG = Char;
	return UART_OK;
}

unsigned int __attribute__((__section__(".SecureSection"))) printstringUart(char *str)
{
	unsigned int k;
	unsigned int x;
	k=UART_OK;
	x=(((unsigned int)(*str))&0x00FF);
	while(x!=0x0000){
		putToUart(x);
		str++;
		x=(((unsigned int)(*str))&0x00FF);
	}
	return k;
}

char __attribute__((__section__(".SecureSection"))) securedispFix(int g)
{
		g&=0x000F;
		if(g>9)g+=0x0007;
		g+=0x0030;
		return g;
}

void __attribute__((__section__(".SecureSection"))) securedisA(int b)
{
	putToUart(securedispFix(b>>4));
	putToUart(securedispFix(b));
}

void __attribute__((__section__(".SecureSection"))) securedisW(int xx)
{
	securedisA(xx>>8);
	securedisA(xx);
}

//*****************************************************************************************************
//
// Flash Memory Access Functions
// Beware: These can *break* and crash your program if used to erase or modify program space!
//
//*****************************************************************************************************
int  __attribute__((__section__(".SecureSection"))) eraseFlashRow(unsigned long address)
{
	// erases one row of 32 instruction words starting at address
	if((writingMode & WRITE_PROTECTION_OFF)!=0)
	{
	if(((address+64)<hexprotectMin)||((address)>hexprotectMax))
	{
		disableAllInterrupts();
		NVMCON=0x4041;
		NVMADRU=(0x00FF & (address>>16));
		NVMADR=(0xFFFF & address);
		__builtin_write_NVM();
		enableAllInterrupts();
		return TRUE;
		} else return FALSE;
	} else return FALSE;
}

unsigned int __attribute__((__section__(".SecureSection"))) computeCRC(unsigned int upto)
{
	unsigned int kl;
	int icrcd;
	unsigned int wordaddress;
	
	disableAllInterrupts();
	wordaddress=0;
	icrcd=0;
	TBLPAG=0;
	addressErrorReporting=0;
	while(wordaddress<upto)
	{
	kl=__builtin_tblrdl(wordaddress);
	icrcd+=(0x00FF & kl);
	icrcd+=(0x00FF & (kl>>8));
	kl=__builtin_tblrdh(wordaddress);
	icrcd+=(0x00FF & kl);
	wordaddress+=2;
	}
	icrcd=-icrcd;
	enableAllInterrupts();
	return (unsigned int)icrcd;
}

void  __attribute__((__section__(".SecureSection"))) readFlashMemory(unsigned long a)
{
	unsigned int i, k;
	unsigned long address;
	unsigned int *p;
	// reads 32 instruction words into RAM
	p=(unsigned int*)&sigCmpx[READING_INDEX];
	address=a;
	addressErrorFlag=READING_VAL;	
	TBLPAG=0x00FF & (address >>16);
	disableAllInterrupts();
	for(i=0; i<32; i++)
	{
	k=__builtin_tblrdl(address);
	*p=k;
	p++;
	k=__builtin_tblrdh(address);
	*p=k;
	p++;
	address+=2;
	}
	addressErrorFlag=0;
	enableAllInterrupts();
}

void __attribute__((interrupt,auto_psv, section(".SecureSection"))) _AddressError(void)
{
	PORTEbits.GBIT=1;
	if(addressErrorReporting==1)
	{	
	    if(addressErrorFlag==READING_VAL){
			    addressErrorsR++;
			    addressErrorFlag=0;
			    printstringUart("ReadAddrErr: 0x");
			    securedisW(hexaddress>>16);
			    securedisW(hexaddress);	
			    putToUart(0x0D);
			    putToUart(0x0A);
		 } else
		 if(addressErrorFlag==WRITING_VAL)
		 {
			 	addressErrorsW++;
			 	addressErrorFlag=0;
		    	printstringUart("WriteAddrErr: 0x");
		    	securedisW(hexaddress>>16);
		    	securedisW(hexaddress);	
		    	putToUart(0x0D);
		    	putToUart(0x0A);
		}
		 
	} else if(addressErrorReporting==0)blink(3);
	INTCON1bits.ADDRERR=0;
}

void  __attribute__((__section__(".SecureSection"))) writeLatchesMemory(unsigned long a)
{
	unsigned int i, k;
	unsigned int *p;
	unsigned long address;
	// writes 32 instruction from RAM into the write latches
	p=(unsigned int*)&sigCmpx[READING_INDEX];
	address=a;
	addressErrorFlag=WRITING_VAL;
	TBLPAG=0x00FF & (address >>16);
	disableAllInterrupts();
	for(i=0; i<32; i++)
	{
	k=*p;
	p++;
	__builtin_tblwtl(address, k);
	k=*p;
	p++;
	__builtin_tblwth(address, k);
	address+=2;
	}
	enableAllInterrupts();
	addressErrorFlag=0;
}

void  __attribute__((__section__(".SecureSection"))) writeFlashMemory(unsigned long address)
{
	unsigned long k;

	k=(address & 0x00007FC0);
	if((k>=0x00000080)&&(k<=0x000000FF))k=FALSE; else k=eraseFlashRow(address);
	if(k==TRUE){
			writeLatchesMemory(address);
			NVMCON=0x4001;	
			NVMADRU=(0x00FF & (address>>16));
			NVMADR=(0xFFFF & address);
			disableAllInterrupts();
			__builtin_write_NVM();
			enableAllInterrupts();
			hextotal+=64;
	} else hexprotectedBytes+=64;
}

unsigned char*  __attribute__((__section__(".SecureSection"))) getASCIIByte(unsigned char* p, unsigned int* result)
{
	// returns 00-FF if ok otherwise above 0x00FF
	unsigned int r;
	unsigned char i;

	*result=0;
	i=*p;
	if((i>='0')&&(i<='9'))i-='0'; else if((i>='A')&&(i<='F'))i=i-'A'+10; else if((i>='a')&&(i<='f'))i=i-'a'+10; else return NULLP;
	r=i<<4;
	p++;
	i=*p;
	if((i>='0')&&(i<='9'))i-='0'; else if((i>='A')&&(i<='F'))i=i-'A'+10; else if((i>='a')&&(i<='f'))i=i-'a'+10; else return NULLP;
	r+=i;
	p++;
	*result=(0x00FF & r);
	return p;
}

unsigned char*  __attribute__((__section__(".SecureSection"))) advanceToNextLine(unsigned char* p)
{
	int i;
	i=0;
	while((i<HEX_LINE_LENGTH_MAX)&&((*p)!=0x00)&&((*p)!=':')){ i++; p++; }
	return p;
}

void __attribute__((__section__(".SecureSection"))) putHexInt(unsigned int xx, unsigned long actualaddress) 
{
	long diff;
	unsigned long internaladd;

	internaladd=(actualaddress>>1);
	diff=internaladd-hexrowaddress;
//	if(diff<0)diff=-diff;					commented this from previous final release version.
	if((diff>=0)&&(diff<64)){				// modified this line too.
		hexpointer=(unsigned int*)&sigCmpx[READING_INDEX];
		hexpointer+=diff;
		*hexpointer=xx; 
		hexbytes+=2;
		} else {
		writeFlashMemory(hexrowaddress);
		hexrowaddress=(internaladd & 0xFFFFFFC0);
		readFlashMemory(hexrowaddress);
		diff=internaladd-hexrowaddress;
		hexpointer=(unsigned int*)&sigCmpx[READING_INDEX];
		hexpointer+=diff;
		*hexpointer=xx;
		hexbytes+=2;
		}
}

unsigned char*  __attribute__((__section__(".SecureSection"))) decodeHexLine(unsigned char* p, unsigned int* error, unsigned int* crc, int writep)
{
	// decode one line of a .hex file starting at p, in ASCII
	// returns the next pointer or NULLP if an error occurred
	unsigned int k;
	unsigned int n, r, s, t, adr;
	*error=HEX_OK;
	k=*p;
	if(k!=':'){ *error=HEX_ERROR; return p; }
	p++;
	*crc=0;
	p=getASCIIByte(p, &n);				// n is the number of bytes in the line!
	if(p==NULLP){ *error=HEX_ERROR; return p; }
	*crc+=n;
	p=getASCIIByte(p, &r);
	if(p==NULLP){ *error=HEX_ERROR; return p; }
	*crc+=r;
	p=getASCIIByte(p, &adr);
	if(p==NULLP){ *error=HEX_ERROR; return p; }
	*crc+=adr;
	adr+=(r<<8);	
	p=getASCIIByte(p, &r);
	if(p==NULLP){ *error=HEX_ERROR; return p; }
	*crc+=r;
	if(r==0x00){
		zerorecords++;
		hexaddress&=0x00FF0000;
		hexaddress+=(0x0000FFFF & adr);
		s=0;
		while(s<n)
		{
		p=getASCIIByte(p, &r);
		if(p==NULLP){ *error=HEX_ERROR; return p; }
		*crc+=r;
		p=getASCIIByte(p, &t);
		if(p==NULLP){ *error=HEX_ERROR; return p; }
		*crc+=t;
		t=t<<8;
		t+=r;
		// all addresses are byte addresses
		if((writep & 0x0001)==0)putHexInt(t, hexaddress);
		hexaddress+=2;
		s+=2;
		}
		p=getASCIIByte(p, &r);
		if(p==NULLP){ *error=HEX_ERROR; return p; }
		*crc=0x0100-((*crc) & 0x00FF);
		if(r!=(0x00FF & (*crc))){ *error=HEX_CRC_ERROR; return p; }
	} else
	if(r==0x01)
	{
		*error=HEX_EOF;
		return p;
	} else
	if(r==0x02)
	{
		tworecords++;
		p=getASCIIByte(p, &r);
		if(p==NULLP){ *error=HEX_ERROR; return p; }
		*crc+=r;
		p=getASCIIByte(p, &t);
		if(p==NULLP){ *error=HEX_ERROR; return p; }
		*crc+=t;
		t=t<<8;
		t+=r;
		hexaddress&=0x0000FFFF;
		hexaddress+=(((unsigned long)t)<<4);
		hexaddress&=0x00FFFFFF;
		p=getASCIIByte(p, &r);	
		if(p==NULLP){ *error=HEX_ERROR; return p; }
		*crc=0x0100-((*crc) & 0x00FF);
		if(r!=(0x00FF & (*crc))){ *error=HEX_CRC_ERROR; return p; }
	} else	
	if(r==0x04)
	{
		fourrecords++;
		p=getASCIIByte(p, &r);
		if(p==NULLP){ *error=HEX_ERROR; return p; }
		*crc+=r;
		p=getASCIIByte(p, &t);
		if(p==NULLP){ *error=HEX_ERROR; return p; }
		*crc+=t;
		t=t<<8;
		t+=r;
		hexaddress&=0x0000FFFF;
		hexaddress+=(((unsigned long)t)<<16);
		hexaddress&=0x00FFFFFF;
		p=getASCIIByte(p, &r);	
		if(p==NULLP){ *error=HEX_ERROR; return p; }
		*crc=0x0100-((*crc) & 0x00FF);
		if(r!=(0x00FF & (*crc))){ *error=HEX_CRC_ERROR; return p; }
	} else
	{
		*error=HEX_ERROR_IN_CODE;
		return p;
	}
	return p;
}

/*
unsigned char*  __attribute__((__section__(".SecureSection"))) decodeHexFile(unsigned char* input, unsigned int* error, unsigned long minprotect, unsigned long maxprotect)
{
	unsigned int err, crc;

	zerorecords=0;
	tworecords=0;
	fourrecords=0;
	hextotal=0;
	hexprotectMin=minprotect & 0xFFFFFFC0;
	hexprotectMax=maxprotect & 0xFFFFFFC0;
	hexaddress=0;
	hexbytes=0;
	hexrowaddress=0x40;
	readFlashMemory(hexrowaddress);
	hexprotectedBytes=0;
	err=HEX_OK;
	while((*input!=0x00)&&(err==HEX_OK))
	{
	input=advanceToNextLine(input);
	input=decodeHexLine(input, &err, &crc, 1);
	}
	*error=err;
	writeFlashMemory(hexrowaddress);
	return input;
}
*/

void __attribute__((__section__(".SecureSection"))) printsecureVersionUart(void)
{
	putToUart(securedispFix(SECURE_VERSION_MAJOR));
	putToUart('.');
	securedisA(SECURE_VERSION_MINOR);	
}

void __attribute__((__section__(".SecureSection"))) enterSecureMode(void)
{
	IEC0&=0x0200;
	IEC1=0;
	IEC2=0;
	PORTEbits.GBIT=1;
	DELAY_STRING;	
	DELAY_STRING;	
	INTCON2bits.ALTIVT=1;
	DELAY_STRING;	
	DELAY_STRING;	
	flushUartFIFO();
}

void __attribute__((__section__(".SecureSection"))) exitSecureMode(void)
{
	_prog_addressT EE_addr;

	INTCON2bits.ALTIVT=0;
	IEC0=0;
	IEC1=0;
	IEC2=0;										// disable All interrupts
  	_init_prog_address(EE_addr, arrayInDataEEPROM);
	EE_addr+=(MAGIC_EEPROM_ADDRESS<<1);
	_erase_eedata(EE_addr, _EE_ROW);
	_wait_eedata();								// delete all the CRC and POR and FIRST BOOT Stuff!
	asm("reset");
}

void __attribute__((__section__(".SecureSection"))) secureIntro(void)
{
	crcerrors=0;
	addressErrorsR=0;
	addressErrorsW=0;
	addressErrorFlag=0;
	addressErrorReporting=1;
	hextotal=0;
	hexprotectMin=HEX_PROTECT_BASE & 0xFFFFFFC0;
	hexprotectMax=HEX_PROTECT_TOP & 0xFFFFFFC0;
	hexprotectedBytes=0;
	hexaddress=0;
	hexbytes=0;
	hexrowaddress=0;
	zerorecords=0;
	tworecords=0;
	fourrecords=0;
	printsecureVersionUart();
	printstringUart("\n");
	printstringUart("Protected 0x");
	securedisW(hexprotectMin);
	printstringUart(" To 0x");
	securedisW(hexprotectMax);
	printstringUart("\nOk.\n");
}

void __attribute__((__section__(".SecureSection"))) secureReport(void)
{
	printstringUart("AddrRead Faults: 0x");
	securedisW(addressErrorsR);
	printstringUart("\n");
	printstringUart("AddrWrite Faults: 0x");
	securedisW(addressErrorsW);
	printstringUart("\n");
	printstringUart("ID0Recs: 0x");
	securedisW(zerorecords);
	printstringUart("\n");
	printstringUart("ID2Recs: 0x");
	securedisW(tworecords);
	printstringUart("\n");
	printstringUart("ID4Recs: 0x");
	securedisW(fourrecords);
	printstringUart("\n");	
	printstringUart("Total(B): 0x");
	securedisW(hexbytes);
	printstringUart("\n");
	printstringUart("Protected(B): 0x");
	securedisW(hexprotectedBytes);
	printstringUart("\n");
	printstringUart("LastAddr: 0x");
	securedisW(hexaddress);
	printstringUart("\n");
	printstringUart("LastRowAddr: 0x");
	securedisW(hexrowaddress);
	printstringUart("\n");
	printstringUart("CRCErrs: 0x");
	securedisW(crcerrors);
	printstringUart("\n");
	printstringUart("System Status: ");
	if(crcerrors==0)printstringUart("Ok.\n"); else printstringUart("Errors.\n");
}

void __attribute__((__section__(".SecureSection"))) secureReset(void)
{
	int i, j;
	printstringUart("Reset in (secs): ");
	for(i=RESET_SECONDS; i>=0; i--)
	{
	putToUart(securedispFix(i));
	printstringUart(" ");
	for(j=0; j<2100; j++)securewait();
	}
	printstringUart("\a\n");
	addressErrorsR=0;
	addressErrorsW=0;
	addressErrorFlag=0;
	addressErrorReporting=0;
}

#if(INCLUDE_FAST_SOFTWARE_UPGRADE==1)

void __attribute__((__section__(".SecureSection"))) securewaitForSync(void)
{ 
	// a sync pattern of 55 AA 19 76
	int l, st;
	st=0;
	while(st<4)
	{
		l=UART_ERROR;
		while(l==UART_ERROR)l=getFromUart();
		switch((st & 0x0003)){
			case 0x00:
				if(l==0x55)st++; else st=0;
				break;	
			case 0x01:
				if(l==0xAA)st++; else st=0;
				break;
			case 0x02:
				if(l==0x19)st++; else st=0;
				break;
			case 0x03:
				if(l==0x76)st++; else st=0;
				break;
		}
	}
}

void __attribute__((__section__(".SecureSection"))) stateFastSoftwareUpgrade(int y)
{ 
	unsigned int wait, cancel, h, l, w1, w2;
	unsigned long j;
	unsigned int* p;
	unsigned long crc;
	
	enterSecureMode();
	printstringUart("\nMusicolour Fast Software Upgrade ver ");
	secureIntro();
	j=0;
	cancel=0;
	p=(unsigned int*)&sigCmpx[READING_INDEX];
	crc=0;
	wait=1;
	while((cancel<CANCEL_TRIES)&&(hexaddress<HEX_PROTECT_BASE))
	{
		if(wait==1){ 
			wait=0; 
			securewaitForSync(); 
			if((writingMode & WRITE_ECHO_ON)!=0)putToUart('S'); 
			}
		l=UART_ERROR;
		while(l==UART_ERROR)
			l=getFromUart();
		crc+=l;
		h=UART_ERROR;
		while(h==UART_ERROR)
			h=getFromUart();
		crc+=h;
		h=h & 0x00FF;
		l=l<<8; 
		l=l & 0xFF00;
		w1=l + h;
		*p++=w1;
		l=UART_ERROR;
		while(l==UART_ERROR)
			l=getFromUart();
		crc+=l;
		h=UART_ERROR;
		while(h==UART_ERROR)
			h=getFromUart();
		crc+=h;
		h=h & 0x00FF;
		l=l<<8; 
		l=l & 0xFF00;
		w2=l + h;
		*p++=w2;
		j++;
		if(j>=32)
			{
			l=UART_ERROR;
			while(l==UART_ERROR)
				l=getFromUart();
			crc=crc & 0x0000FF;
			crc=0x00000100-crc;
			if((0x0000FF & l)==(crc & 0x00000FF))
				{	
				hextotal+=64;
				hexbytes+=64;	
				hexaddress+=32;
				putToUart(l);
				putToUart(crc);
				cancel=0;
				} else 
				{
				cancel++;
				putToUart(l);
				putToUart(crc);
				}
				j=0;
				wait=1;
				p=(unsigned int*)&sigCmpx[READING_INDEX];
			}
	}
	hexrowaddress=hexaddress;
	secureReport();
	secureReset();
	exitSecureMode();
}
#else
void __attribute__((__section__(".SecureSection"))) stateFastSoftwareUpgrade(int y)
{ 
	enterSecureMode();
	printstringUart("\nMusicolour Fast Software Upgrade Not Implemented.  ");
	secureReset();
	exitSecureMode();
}
#endif
	
void __attribute__((__section__(".SecureSection"))) stateSoftwareUpgrade(int y)
{ 
	unsigned int i, l, err;
	unsigned long j;
	unsigned char *input, *tempp;
	unsigned int crc;

	enterSecureMode();
	printstringUart("\nMusicolour Software Upgrade ver ");
	secureIntro();
	readFlashMemory(hexrowaddress);
	hexprotectedBytes=0;
	input=(unsigned char*)&sigCmpx[WRITING_INDEX];
	i=0;
	j=0;
	l=0;
	err=HEX_OK;
	while((err!=HEX_EOF)&&(securenoKeys()))
	{
		i=UART_ERROR;
		while(i==UART_ERROR)i=getFromUart();
		//putToUart(i);
		l++;
		j++;
		*input=i; 
		input++;
		if((i==0x0000)||(i==0x000D)||(i==0x000A)||(l>=HEX_LINE_LENGTH_MAX))
		{
			*input=0x0000;
			if((writingMode & WRITE_ECHO_ON)!=0)
			{ printstringUart("\n>Received Line: "); printstringUart((char*)&sigCmpx[WRITING_INDEX]); printstringUart("\n");	}
			err=HEX_OK;
			input=(unsigned char*)&sigCmpx[WRITING_INDEX];
			while(((*input)!=0x00)&&(err==HEX_OK))
			{
			input=advanceToNextLine(input);
			tempp=decodeHexLine(input, &err, &crc, 1);
			if(err!=HEX_EOF)
				{
				if(err==HEX_CRC_ERROR)crcerrors++;
				if((writingMode & WRITE_ECHO_CRC)!=0)
				{	
				if(err==HEX_OK){ printstringUart(" CRC: 0x"); securedisA(crc); printstringUart(" Ok.\n"); }  else if(err==HEX_CRC_ERROR){ printstringUart(" CRC: 0x"); securedisA(crc); printstringUart(" Bad.\n"); }
				}
				if(err==HEX_OK)
				{
				if((writingMode & WRITE_PROTECTION_OFF)!=0)tempp=decodeHexLine(input, &err, &crc, 0);
				}
				input=tempp;
				}
			}
			l=0;
			input=(unsigned char*)&sigCmpx[WRITING_INDEX];
		} 	
	}
	// clean up and end!
	if((writingMode & WRITE_PROTECTION_OFF)!=0)writeFlashMemory(hexrowaddress);
	printstringUart("EOF.\n");
	secureReport();
	secureReset();
	exitSecureMode();
}

//*****************************************************************************
// End of Secure Sections
//*****************************************************************************
